<?php

namespace App\Jobs;

use App\Models\Order;
use App\Services\EmailService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SendOrderEmailJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 60;

    /**
     * The number of seconds to wait before retrying the job.
     *
     * @var int
     */
    public $backoff = 30;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public Order $order
    ) {}

    /**
     * Execute the job.
     */
    public function handle(EmailService $emailService): void
    {
        try {
            Log::info("Sending order email", [
                'order_id' => $this->order->id,
                'email' => $this->order->email,
            ]);

            // Wait for PDF if it's not ready yet
            $this->waitForPDF();

            $success = $emailService->sendOrderEmail($this->order);

            if ($success) {
                $this->order->markAsCompleted();
                Log::info("Order email sent successfully", ['order_id' => $this->order->id]);
            } else {
                throw new \Exception("Email service returned false");
            }

        } catch (\Exception $e) {
            Log::error("Email sending failed", [
                'order_id' => $this->order->id,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts(),
            ]);

            if ($this->attempts() >= $this->tries) {
                $this->order->markAsFailed("Email sending failed: " . $e->getMessage());

                // Optionally send failure notification
                // $emailService->sendOrderFailureNotification($this->order, $e->getMessage());
            }

            throw $e;
        }
    }

    /**
     * Wait for PDF to be generated (max 30 seconds).
     */
    protected function waitForPDF(): void
    {
        $maxWait = 30; // seconds
        $waited = 0;

        while (!$this->order->pdf && $waited < $maxWait) {
            sleep(2);
            $waited += 2;
            $this->order->refresh();
        }

        if (!$this->order->pdf) {
            Log::warning("PDF not ready after waiting", [
                'order_id' => $this->order->id,
                'waited' => $waited,
            ]);
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Email job failed after all retries", [
            'order_id' => $this->order->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
