<?php

namespace App\Providers;

use Illuminate\Foundation\Support\Providers\AuthServiceProvider as ServiceProvider;
use Illuminate\Support\Facades\Gate;
use App\Models\Company;
use App\Models\Pharmacy;
use App\Models\Product;
use App\Models\Order;
use App\Policies\CompanyPolicy;
use App\Policies\PharmacyPolicy;
use App\Policies\ProductPolicy;
use App\Policies\OrderPolicy;

/**
 * AuthServiceProvider
 *
 * Registers authorization policies for multi-tenant access control.
 * All policies enforce strict data isolation to prevent cross-company access.
 *
 * Policy Mapping:
 * - Company → CompanyPolicy: Controls company management access
 * - Pharmacy → PharmacyPolicy: Ensures pharmacies are company-scoped
 * - Product → ProductPolicy: Ensures products are company-scoped
 * - Order → OrderPolicy: Ensures orders are company-scoped
 */
class AuthServiceProvider extends ServiceProvider
{
    /**
     * The model to policy mappings for the application.
     *
     * Maps Eloquent models to their corresponding authorization policies.
     * Laravel will automatically discover and apply these policies when
     * using $this->authorize() in controllers or Gate::allows() in code.
     *
     * @var array<class-string, class-string>
     */
    protected $policies = [
        Company::class => CompanyPolicy::class,
        Pharmacy::class => PharmacyPolicy::class,
        Product::class => ProductPolicy::class,
        Order::class => OrderPolicy::class,
    ];

    /**
     * Register any authentication / authorization services.
     */
    public function boot(): void
    {
        $this->registerPolicies();

        // Define custom gates for role-based access control

        /**
         * Gate: isSuperAdmin
         * Determines if the user has super admin privileges.
         */
        Gate::define('isSuperAdmin', function ($user) {
            return $user->role === 'super_admin';
        });

        /**
         * Gate: isCompanyAdmin
         * Determines if the user is a company admin (or higher).
         */
        Gate::define('isCompanyAdmin', function ($user) {
            return in_array($user->role, ['super_admin', 'company_admin']);
        });

        /**
         * Gate: isRep
         * Determines if the user is a sales rep.
         */
        Gate::define('isRep', function ($user) {
            return $user->role === 'rep';
        });

        /**
         * Gate: belongsToCompany
         * Determines if a model belongs to the user's company.
         * CRITICAL: This gate enforces multi-tenant data isolation.
         *
         * Usage: Gate::allows('belongsToCompany', $model)
         */
        Gate::define('belongsToCompany', function ($user, $model) {
            // Super admin can access all companies
            if ($user->role === 'super_admin') {
                return true;
            }

            // Check if model has company_id attribute
            if (!property_exists($model, 'company_id')) {
                return false;
            }

            // CRITICAL: Verify company_id matches
            return $user->company_id === $model->company_id;
        });

        /**
         * Gate: manageUsers
         * Determines if the user can manage other users.
         * Super admins can manage all users.
         * Company admins can manage users in their company.
         */
        Gate::define('manageUsers', function ($user, $targetUser = null) {
            // Super admin can manage all users
            if ($user->role === 'super_admin') {
                return true;
            }

            // Company admin can manage users in their company
            if ($user->role === 'company_admin') {
                // If no target user specified, allow (for index/create)
                if (!$targetUser) {
                    return true;
                }

                // Verify target user belongs to same company
                return $user->company_id === $targetUser->company_id;
            }

            // Reps cannot manage users
            return false;
        });

        /**
         * Gate: viewAnalytics
         * Determines if the user can view analytics and reports.
         * Super admins and company admins can view analytics.
         */
        Gate::define('viewAnalytics', function ($user) {
            return in_array($user->role, ['super_admin', 'company_admin']);
        });

        /**
         * Gate: manageCatalog
         * Determines if the user can manage product catalog.
         * Only super admins and company admins can modify catalog.
         */
        Gate::define('manageCatalog', function ($user) {
            return in_array($user->role, ['super_admin', 'company_admin']);
        });
    }
}
