<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\Wholesaler;
use App\Models\PharmacyWholesalerAccount;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class WholesalerController extends Controller
{
    /**
     * Display a listing of wholesalers used by supplier's reps (read-only).
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Get all unique wholesalers used by pharmacies belonging to this supplier's company
        $query = Wholesaler::query()
            ->whereHas('pharmacies', function ($q) use ($companyId) {
                $q->where('pharmacies.company_id', $companyId);
            });

        $wholesalers = $query
            ->withCount([
                'pharmacies as pharmacies_count' => function ($q) use ($companyId) {
                    $q->where('pharmacies.company_id', $companyId);
                },
                'orders as orders_count' => function ($q) use ($companyId) {
                    $q->where('orders.company_id', $companyId);
                }
            ])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('wholesaler_name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->orderBy('wholesaler_name')
            ->paginate(15)
            ->withQueryString();

        // Get pharmacy-wholesaler account details for the current company
        $wholesalers->getCollection()->transform(function ($wholesaler) use ($companyId) {
            $wholesaler->company_pharmacy_accounts = PharmacyWholesalerAccount::query()
                ->whereHas('pharmacy', function ($q) use ($companyId) {
                    $q->where('company_id', $companyId);
                })
                ->where('wholesaler_id', $wholesaler->id)
                ->with('pharmacy:id,pharmacy_name')
                ->get();

            return $wholesaler;
        });

        return Inertia::render('SupplierAdmin/Wholesalers/Index', [
            'wholesalers' => $wholesalers,
            'filters' => $request->only(['search', 'status']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('dashboard')],
                ['label' => 'Wholesalers', 'url' => null],
            ],
        ]);
    }
}
