<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\Pharmacy;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class PharmacyController extends Controller
{
    /**
     * Display a listing of pharmacies created by supplier's reps (read-only).
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Get all pharmacies created by reps from this supplier's company
        $query = Pharmacy::where('company_id', $companyId);

        $pharmacies = $query
            ->with(['createdByRep', 'wholesalers'])
            ->withCount('orders')
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('pharmacy_name', 'like', "%{$search}%")
                        ->orWhere('registration_number', 'like', "%{$search}%")
                        ->orWhere('contact_person', 'like', "%{$search}%");
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('rep_id'), function ($q, $repId) {
                $q->where('created_by_rep_id', $repId);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get filter options scoped to supplier's company
        $reps = User::where('role', 'rep')
            ->where('company_id', $companyId)
            ->orderBy('name')
            ->get(['id', 'name']);

        return Inertia::render('SupplierAdmin/Pharmacies/Index', [
            'pharmacies' => $pharmacies,
            'reps' => $reps,
            'filters' => $request->only(['search', 'status', 'rep_id']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('dashboard')],
                ['label' => 'Pharmacies', 'url' => null],
            ],
        ]);
    }

    /**
     * Display the specified pharmacy with details and order history (read-only).
     */
    public function show(Request $request, Pharmacy $pharmacy): Response
    {
        $user = $request->user();

        // Ensure pharmacy belongs to supplier's company
        if ($pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $pharmacy->load([
            'createdByRep',
            'wholesalers' => function ($query) {
                $query->withPivot(['account_number', 'is_primary', 'status']);
            },
        ]);

        // Get order history for this pharmacy
        $orders = $pharmacy->orders()
            ->with(['rep', 'wholesaler'])
            ->latest()
            ->paginate(10);

        return Inertia::render('SupplierAdmin/Pharmacies/Show', [
            'pharmacy' => $pharmacy,
            'orders' => $orders,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('dashboard')],
                ['label' => 'Pharmacies', 'url' => route('supplier-admin.pharmacies.index')],
                ['label' => $pharmacy->pharmacy_name, 'url' => null],
            ],
        ]);
    }
}
