<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreProductRequest;
use App\Http\Resources\ProductResource;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class ProductController extends Controller
{
    /**
     * Display a listing of products.
     */
    public function index(Request $request): AnonymousResourceCollection
    {
        // Authorize: check if user can view products
        $this->authorize('viewAny', Product::class);

        $query = Product::query()->with('brand');

        // Filter by brand
        if ($request->has('brand_id')) {
            $query->byBrand($request->brand_id);
        }

        // Filter by active status
        if ($request->has('active')) {
            $query->where('active', filter_var($request->active, FILTER_VALIDATE_BOOLEAN));
        }

        // Search products
        if ($request->has('search')) {
            $query->search($request->search);
        }

        // Sorting
        $sortField = $request->input('sort', 'sort_order');
        $sortDirection = $request->input('direction', 'asc');
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        if ($request->has('per_page')) {
            $products = $query->paginate($request->per_page);
            return ProductResource::collection($products);
        }

        $products = $query->get();
        return ProductResource::collection($products);
    }

    /**
     * Store a newly created product.
     */
    public function store(StoreProductRequest $request): JsonResponse
    {
        // Authorize: only company_admin can create products
        $this->authorize('create', Product::class);

        $data = $request->validated();
        $data['company_id'] = auth()->user()->company_id;

        $product = Product::create($data);
        $product->load('brand');

        return response()->json([
            'success' => true,
            'message' => 'Product created successfully.',
            'data' => new ProductResource($product),
        ], 201);
    }

    /**
     * Display the specified product.
     */
    public function show(Product $product): JsonResponse
    {
        // Authorize: check if user can view this product (company-scoped)
        $this->authorize('view', $product);

        $product->load('brand');

        return response()->json([
            'success' => true,
            'data' => new ProductResource($product),
        ]);
    }

    /**
     * Update the specified product.
     */
    public function update(Request $request, Product $product): JsonResponse
    {
        // Authorize: only company_admin can update products
        $this->authorize('update', $product);

        $validated = $request->validate([
            'brand_id' => ['sometimes', 'integer', 'exists:brands,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'size' => ['nullable', 'string', 'max:100'],
            'price' => ['nullable', 'numeric', 'min:0', 'max:99999999.99'],
            'barcode' => ['sometimes', 'string', 'max:255', 'unique:products,barcode,' . $product->id],
            'active' => ['sometimes', 'boolean'],
            'sort_order' => ['sometimes', 'integer', 'min:0'],
        ]);

        $product->update($validated);
        $product->load('brand');

        return response()->json([
            'success' => true,
            'message' => 'Product updated successfully.',
            'data' => new ProductResource($product),
        ]);
    }

    /**
     * Remove the specified product.
     */
    public function destroy(Product $product): JsonResponse
    {
        // Authorize: only company_admin can delete products
        $this->authorize('delete', $product);

        $product->delete();

        return response()->json([
            'success' => true,
            'message' => 'Product deleted successfully.',
        ]);
    }

    /**
     * Get products by brand name (for legacy compatibility).
     */
    public function byBrandName(Request $request, string $brandName): AnonymousResourceCollection
    {
        $products = Product::query()
            ->whereHas('brand', function ($q) use ($brandName) {
                $q->where('name', 'like', "%{$brandName}%");
            })
            ->with('brand')
            ->where('active', true)
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return ProductResource::collection($products);
    }
}
