<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreOrderRequest;
use App\Http\Requests\UpdateOrderRequest;
use App\Http\Resources\OrderResource;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Pharmacy;
use App\Jobs\SendOrderEmailJob;
use App\Jobs\GeneratePDFJob;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    /**
     * Display a listing of orders.
     */
    public function index(Request $request): AnonymousResourceCollection
    {
        $query = Order::query()->with(['items.product.brand', 'rep', 'pharmacy', 'wholesaler']);

        // Filter by status
        if ($request->has('status')) {
            $query->byStatus($request->status);
        }

        // Filter by pharmacy_id
        if ($request->has('pharmacy_id')) {
            $query->byPharmacy($request->pharmacy_id);
        }

        // Filter by rep_id
        if ($request->has('rep_id')) {
            $query->byRep($request->rep_id);
        }

        // Filter by wholesaler_id
        if ($request->has('wholesaler_id')) {
            $query->byWholesaler($request->wholesaler_id);
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->dateRange($request->start_date, $request->end_date);
        }

        // Sorting
        $sortField = $request->input('sort', 'created_at');
        $sortDirection = $request->input('direction', 'desc');
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $orders = $query->paginate($perPage);

        return OrderResource::collection($orders);
    }

    /**
     * Store a newly created order.
     */
    public function store(UpdateOrderRequest $request): JsonResponse
    {
        try {
            DB::beginTransaction();

            // Auto-assign company_id and rep_id from authenticated user
            $user = auth()->user();

            // Validate pharmacy and wholesaler belong to rep's company
            $pharmacy = Pharmacy::findOrFail($request->pharmacy_id);

            // Get pharmacy email for order
            $pharmacyEmail = $pharmacy->email ?? $user->email;

            // Create order with auto-generated order_number (via model event)
            $order = Order::create([
                'company_id' => $user->company_id,
                'rep_id' => $user->id,
                'pharmacy_id' => $request->pharmacy_id,
                'wholesaler_id' => $request->wholesaler_id,
                'wholesaler_account_number' => $request->wholesaler_account_number,
                'delivery_date' => $request->delivery_date,
                'notes' => $request->notes,
                'status' => 'pending',
            ]);

            // Create order items
            foreach ($request->items as $item) {
                $orderItem = OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                ]);

                // Create product snapshot for historical accuracy
                $orderItem->createProductSnapshot();
            }

            // Update total items and total amount
            $order->updateTotalItems();
            // Note: updateTotalAmount() would require Product model to have price field
            // For now, we'll leave it as 0 or calculate it based on snapshot data if available

            // Dispatch jobs for PDF generation and email sending
            GeneratePDFJob::dispatch($order);
            SendOrderEmailJob::dispatch($order);

            DB::commit();

            // Reload relationships
            $order->load(['items.product.brand', 'pdf', 'rep', 'pharmacy', 'wholesaler']);

            return response()->json([
                'success' => true,
                'message' => 'Order created successfully. Email will be sent shortly.',
                'data' => new OrderResource($order),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create order.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified order.
     */
    public function show(Order $order): JsonResponse
    {
        // The global scope will automatically ensure the order belongs to the user's company
        $order->load(['items.product.brand', 'pdf', 'rep', 'pharmacy', 'wholesaler']);

        return response()->json([
            'success' => true,
            'data' => new OrderResource($order),
        ]);
    }

    /**
     * Update the specified order status.
     */
    public function updateStatus(Request $request, Order $order): JsonResponse
    {
        $validated = $request->validate([
            'status' => ['required', 'in:pending,processing,completed,failed'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        $order->update($validated);

        if ($validated['status'] === 'completed') {
            $order->markAsCompleted();
        }

        return response()->json([
            'success' => true,
            'message' => 'Order status updated successfully.',
            'data' => new OrderResource($order),
        ]);
    }

    /**
     * Resend order email.
     */
    public function resendEmail(Order $order): JsonResponse
    {
        SendOrderEmailJob::dispatch($order);

        return response()->json([
            'success' => true,
            'message' => 'Order email queued for sending.',
        ]);
    }

    /**
     * Regenerate order PDF.
     */
    public function regeneratePDF(Order $order): JsonResponse
    {
        GeneratePDFJob::dispatch($order);

        return response()->json([
            'success' => true,
            'message' => 'PDF generation queued.',
        ]);
    }
}
