<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard with statistics and recent activity.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;
        $isSuperAdmin = $user->role === 'super_admin';

        // Base query for company scoping
        $ordersQuery = $isSuperAdmin
            ? Order::query()
            : Order::where('company_id', $companyId);

        $productsQuery = $isSuperAdmin
            ? Product::query()
            : Product::where('company_id', $companyId);

        $repsQuery = $isSuperAdmin
            ? User::where('role', 'rep')
            : User::where('role', 'rep')->where('company_id', $companyId);

        // Date ranges
        $today = Carbon::today();
        $weekStart = Carbon::now()->startOfWeek();
        $monthStart = Carbon::now()->startOfMonth();

        // Order statistics
        $stats = [
            'orders_today' => (clone $ordersQuery)->whereDate('created_at', $today)->count(),
            'orders_week' => (clone $ordersQuery)->where('created_at', '>=', $weekStart)->count(),
            'orders_month' => (clone $ordersQuery)->where('created_at', '>=', $monthStart)->count(),
            'total_orders' => (clone $ordersQuery)->count(),

            'active_reps' => (clone $repsQuery)->where('status', 'active')->count(),
            'total_reps' => (clone $repsQuery)->count(),

            'active_products' => (clone $productsQuery)->where('active', true)->count(),
            'total_products' => (clone $productsQuery)->count(),

            'pending_orders' => (clone $ordersQuery)->where('status', 'pending')->count(),
            'processing_orders' => (clone $ordersQuery)->where('status', 'processing')->count(),
            'completed_orders' => (clone $ordersQuery)->where('status', 'completed')->count(),
            'failed_orders' => (clone $ordersQuery)->where('status', 'failed')->count(),
        ];

        // Super Admin specific stats
        if ($isSuperAdmin) {
            $stats['total_companies'] = \App\Models\Company::count();
            $stats['active_companies'] = \App\Models\Company::where('status', 'active')->count();
            $stats['total_pharmacies'] = \App\Models\Pharmacy::count();
            $stats['companies_added_month'] = \App\Models\Company::where('created_at', '>=', $monthStart)->count();
        }

        // Revenue statistics (based on total items as proxy)
        $revenue = [
            'today' => (clone $ordersQuery)->whereDate('created_at', $today)->sum('total_items'),
            'week' => (clone $ordersQuery)->where('created_at', '>=', $weekStart)->sum('total_items'),
            'month' => (clone $ordersQuery)->where('created_at', '>=', $monthStart)->sum('total_items'),
        ];

        // Recent orders
        $recentOrders = (clone $ordersQuery)
            ->with(['items.product', 'pharmacy', 'user'])
            ->latest()
            ->limit(10)
            ->get();

        // Top performing reps (by order count)
        $topReps = (clone $repsQuery)
            ->withCount(['orders' => function ($query) use ($monthStart) {
                $query->where('created_at', '>=', $monthStart);
            }])
            ->orderBy('orders_count', 'desc')
            ->limit(5)
            ->get();

        // Top ordering pharmacies (by order count this month)
        $topPharmaciesQuery = $isSuperAdmin
            ? Order::query()
            : Order::where('company_id', $companyId);

        $topPharmacies = $topPharmaciesQuery
            ->select('pharmacy_id')
            ->with('pharmacy')
            ->where('created_at', '>=', $monthStart)
            ->groupBy('pharmacy_id')
            ->selectRaw('pharmacy_id, COUNT(*) as order_count, SUM(total_items) as total_items')
            ->orderBy('order_count', 'desc')
            ->limit(5)
            ->get();

        // Recent companies (Super Admin only)
        $recentCompanies = null;
        if ($isSuperAdmin) {
            $recentCompanies = \App\Models\Company::withCount(['users', 'products', 'orders'])
                ->latest()
                ->limit(5)
                ->get();
        }

        return Inertia::render('Admin/Dashboard', [
            'stats' => $stats,
            'revenue' => $revenue,
            'recentOrders' => $recentOrders,
            'topReps' => $topReps,
            'topPharmacies' => $topPharmacies,
            'recentCompanies' => $recentCompanies,
            'isSuperAdmin' => $isSuperAdmin,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => null],
            ],
        ]);
    }
}
