# Authorization & Multi-Tenant Security Implementation

This document describes the comprehensive authorization policies and middleware implemented for multi-tenant data isolation in the Haleon API.

## Overview

The system implements **strict multi-tenant data isolation** using Laravel's authorization policies and custom middleware. Each company's data is completely isolated, and users can ONLY access data belonging to their company.

## Security Architecture

### 1. Role-Based Access Control (RBAC)

Three user roles are defined in the system:

#### super_admin
- Can access ALL data across ALL companies
- Can create, update, and delete companies
- Full system administration rights
- Bypasses company-scoped restrictions

#### company_admin
- Can manage all data within their company
- Can create/update/delete pharmacies, products, and orders
- Can manage users within their company
- Cannot access other companies' data

#### rep (Sales Representative)
- Can view products and pharmacies from their company
- Can create pharmacies and orders
- Can update only their own orders
- Cannot delete data or modify product catalog
- Cannot access other companies' data

### 2. Authorization Policies

Four comprehensive policies enforce role-based permissions:

#### CompanyPolicy
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/CompanyPolicy.php`

- `viewAny()` - Only super_admin can view all companies
- `view()` - Users can view their own company
- `create()` - Only super_admin
- `update()` - Only super_admin or company_admin of that company
- `delete()` - Only super_admin

#### PharmacyPolicy
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/PharmacyPolicy.php`

- `viewAny()` - All authenticated users (scoped to their company)
- `view()` - User can view if pharmacy belongs to their company
- `create()` - Any authenticated user (auto-scoped to their company)
- `update()` - User can update if pharmacy belongs to their company
- `delete()` - Only company_admin can delete

#### ProductPolicy
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/ProductPolicy.php`

- `viewAny()` - All authenticated users (scoped to their company)
- `view()` - User can view if product belongs to their company
- `create()` - Only company_admin
- `update()` - Only company_admin
- `delete()` - Only company_admin

#### OrderPolicy
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/OrderPolicy.php`

- `viewAny()` - All authenticated users (scoped to their company)
- `view()` - User can view if order belongs to their company
- `create()` - Any authenticated user from same company
- `update()` - Only order creator (rep) or company_admin
- `delete()` - Only company_admin
- `resendEmail()` - Order creator or company_admin
- `regeneratePDF()` - Order creator or company_admin

### 3. Custom Middleware

#### EnsureCompanyAccess
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Middleware/EnsureCompanyAccess.php`

**Purpose:** Provides an additional security layer beyond policies to verify model access.

**How it works:**
- Checks route parameters for company-scoped models (Pharmacy, Product, Order, Company)
- Verifies that the model's `company_id` matches the authenticated user's `company_id`
- Super admins bypass this check
- Returns 403 Forbidden if unauthorized

**Usage:**
```php
Route::middleware(['auth:sanctum', 'ensure.company.access'])->group(function () {
    Route::apiResource('pharmacies', PharmacyController::class);
});
```

#### EnsureActiveUser
**Location:** `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Middleware/EnsureActiveUser.php`

**Purpose:** Blocks inactive users from accessing the system.

**How it works:**
- Verifies user's status field is 'active'
- Logs blocked access attempts
- Returns 403 Forbidden with helpful message if inactive

**Usage:**
```php
Route::middleware(['auth:sanctum', 'ensure.active.user'])->group(function () {
    // Protected routes
});
```

### 4. Global Scopes for Automatic Filtering

Several models include global scopes that automatically filter queries by `company_id`:

#### Pharmacy Model
```php
protected static function booted(): void
{
    static::addGlobalScope('company', function (Builder $query) {
        if (auth()->check() && auth()->user()->company_id) {
            $query->where('company_id', auth()->user()->company_id);
        }
    });
}
```

#### Product Model
Same pattern - automatically filters products by company_id.

#### Order Model
Same pattern - automatically filters orders by company_id.

#### User Model
Filters users by company_id (except for super_admin).

**Important:** Global scopes work automatically but can be bypassed using `withoutGlobalScope('company')` if needed (use with caution).

## Implementation Guide

### Using Policies in Controllers

Authorization checks should be added at the beginning of controller methods:

```php
public function index(Request $request)
{
    // Check if user can view any pharmacies
    $this->authorize('viewAny', Pharmacy::class);

    // Query is automatically scoped to user's company by global scope
    $pharmacies = Pharmacy::all();

    return PharmacyResource::collection($pharmacies);
}

public function show(Pharmacy $pharmacy)
{
    // Check if user can view this specific pharmacy
    $this->authorize('view', $pharmacy);

    return new PharmacyResource($pharmacy);
}

public function store(Request $request)
{
    // Check if user can create pharmacies
    $this->authorize('create', Pharmacy::class);

    // Auto-assign company_id from authenticated user
    $data = $request->validated();
    $data['company_id'] = auth()->user()->company_id;

    $pharmacy = Pharmacy::create($data);

    return new PharmacyResource($pharmacy);
}

public function update(Request $request, Pharmacy $pharmacy)
{
    // Check if user can update this pharmacy
    $this->authorize('update', $pharmacy);

    $pharmacy->update($request->validated());

    return new PharmacyResource($pharmacy);
}

public function destroy(Pharmacy $pharmacy)
{
    // Check if user can delete this pharmacy (only company_admin)
    $this->authorize('delete', $pharmacy);

    $pharmacy->delete();

    return response()->json(['message' => 'Deleted successfully']);
}
```

### Using Gates

Custom gates are defined in AuthServiceProvider:

```php
// Check if user is super admin
if (Gate::allows('isSuperAdmin')) {
    // Perform super admin action
}

// Check if user is company admin or higher
if (Gate::allows('isCompanyAdmin')) {
    // Perform admin action
}

// Check if model belongs to user's company
if (Gate::allows('belongsToCompany', $model)) {
    // Access allowed
}

// Check if user can manage another user
if (Gate::allows('manageUsers', $targetUser)) {
    // Manage user
}

// Check if user can view analytics
if (Gate::allows('viewAnalytics')) {
    // Show analytics
}

// Check if user can manage product catalog
if (Gate::allows('manageCatalog')) {
    // Manage catalog
}
```

### Applying Middleware to Routes

In `routes/api.php`:

```php
// Protected API routes with full security
Route::middleware(['auth:sanctum', 'ensure.active.user', 'ensure.company.access'])
    ->prefix('v1')
    ->group(function () {

        // Pharmacy routes
        Route::apiResource('pharmacies', PharmacyController::class);

        // Product routes
        Route::apiResource('products', ProductController::class);

        // Order routes
        Route::apiResource('orders', OrderController::class);
        Route::post('orders/{order}/resend-email', [OrderController::class, 'resendEmail']);
        Route::post('orders/{order}/regenerate-pdf', [OrderController::class, 'regeneratePDF']);
    });

// Super admin only routes
Route::middleware(['auth:sanctum', 'ensure.active.user'])
    ->prefix('admin')
    ->group(function () {

        Route::apiResource('companies', CompanyController::class);
    });
```

## Data Flow Example

### Creating an Order (Rep User)

1. **Request arrives:** `POST /api/v1/orders`

2. **Middleware chain executes:**
   - `auth:sanctum` - Verifies user is authenticated
   - `ensure.active.user` - Verifies user status is 'active'
   - `ensure.company.access` - (Skipped for create, no route parameter)

3. **Controller authorization:**
   ```php
   $this->authorize('create', Order::class);
   ```
   - OrderPolicy checks: All authenticated users can create orders

4. **Order creation:**
   ```php
   $order = Order::create([
       'company_id' => auth()->user()->company_id,  // Auto-assigned
       'rep_id' => auth()->id(),                     // Auto-assigned
       'pharmacy_id' => $request->pharmacy_id,
       // ... other fields
   ]);
   ```

5. **Result:** Order is created and scoped to the rep's company

### Viewing an Order (Different User)

1. **Request arrives:** `GET /api/v1/orders/123`

2. **Route model binding:** Laravel fetches Order #123
   - Global scope automatically filters: `WHERE company_id = {user's company_id}`
   - If order doesn't belong to user's company, returns 404

3. **Middleware chain executes:**
   - `auth:sanctum` - Verifies authentication
   - `ensure.active.user` - Verifies active status
   - `ensure.company.access` - Verifies order's `company_id` matches user's `company_id`
   - Returns 403 if mismatch

4. **Controller authorization:**
   ```php
   $this->authorize('view', $order);
   ```
   - OrderPolicy checks: User's `company_id` must match order's `company_id`

5. **Result:** Order is returned only if all checks pass

## Security Best Practices

### 1. Always Auto-Assign company_id

When creating records, never trust client-provided `company_id`:

```php
// GOOD ✓
$data = $request->validated();
$data['company_id'] = auth()->user()->company_id;
$product = Product::create($data);

// BAD ✗
$product = Product::create($request->validated()); // Client could inject company_id
```

### 2. Use Both Policies and Middleware

Policies handle **what** a user can do.
Middleware handles **cross-cutting concerns** like company access verification.

Both should be used together for defense in depth.

### 3. Never Bypass Global Scopes Without Good Reason

```php
// AVOID unless absolutely necessary
$allProducts = Product::withoutGlobalScope('company')->get();

// PREFERRED - scope is automatically applied
$companyProducts = Product::all();
```

### 4. Log Authorization Failures

The `EnsureActiveUser` middleware logs blocked access:

```php
\Log::warning('Inactive user attempted to access the system', [
    'user_id' => $user->id,
    'email' => $user->email,
    'status' => $user->status,
    'ip' => $request->ip(),
    'route' => $request->path(),
]);
```

Consider adding similar logging to policies for security auditing.

### 5. Test Authorization Thoroughly

Always test with users from different companies:

```php
// Feature test example
public function test_user_cannot_view_other_company_pharmacy()
{
    $company1 = Company::factory()->create();
    $company2 = Company::factory()->create();

    $user1 = User::factory()->for($company1)->create();
    $pharmacy2 = Pharmacy::factory()->for($company2)->create();

    $response = $this->actingAs($user1)
        ->getJson("/api/v1/pharmacies/{$pharmacy2->id}");

    $response->assertStatus(404); // Global scope filters it out
}
```

## Testing Checklist

- [ ] Super admin can access all companies' data
- [ ] Company admin can manage their company's data
- [ ] Company admin cannot access other companies' data
- [ ] Rep can view products and pharmacies from their company
- [ ] Rep can create orders and pharmacies
- [ ] Rep cannot delete pharmacies or products
- [ ] Rep cannot update other reps' orders
- [ ] Inactive users receive 403 Forbidden
- [ ] Cross-company access attempts return 403 or 404
- [ ] Global scopes correctly filter queries

## Troubleshooting

### "This action is unauthorized" Error

**Cause:** Policy check failed.

**Solution:** Verify user has correct role and model belongs to their company.

### Getting 404 Instead of Record

**Cause:** Global scope filtered out the record.

**Solution:** Verify the record's `company_id` matches user's `company_id`.

### Middleware Not Applied

**Cause:** Middleware not registered or route not in middleware group.

**Solution:** Check `bootstrap/app.php` for middleware registration and route group setup.

## Files Created

### Policies
- `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/CompanyPolicy.php`
- `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/PharmacyPolicy.php`
- `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/ProductPolicy.php`
- `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/OrderPolicy.php`

### Middleware
- `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Middleware/EnsureCompanyAccess.php`
- `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Middleware/EnsureActiveUser.php`

### Service Providers
- `/home/orderlyxdev/public_html/HaleonAPI/app/Providers/AuthServiceProvider.php`

### Updated Files
- `/home/orderlyxdev/public_html/HaleonAPI/bootstrap/app.php` (middleware registration)
- `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Controllers/Api/PharmacyController.php` (authorization examples)
- `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Controllers/Api/ProductController.php` (authorization examples)

## Summary

This implementation provides **multi-layered security** for multi-tenant data isolation:

1. **Global Scopes** - Automatic query filtering by company_id
2. **Authorization Policies** - Role-based permission checks
3. **Custom Middleware** - Additional company access verification
4. **Auto-assignment** - Prevents company_id tampering
5. **Comprehensive Gates** - Reusable authorization logic

All layers work together to ensure **ZERO cross-company data leakage**.
