<?php

use App\Http\Controllers\Api\BrandController;
use App\Http\Controllers\Api\ProductController;
use App\Http\Controllers\Api\OrderController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// API Version 1
Route::prefix('v1')->group(function () {

    // Public routes (no authentication required initially)
    Route::prefix('brands')->group(function () {
        Route::get('/', [BrandController::class, 'index']);
        Route::get('/{brand}', [BrandController::class, 'show']);
        Route::post('/', [BrandController::class, 'store']);
        Route::put('/{brand}', [BrandController::class, 'update']);
        Route::delete('/{brand}', [BrandController::class, 'destroy']);
    });

    Route::prefix('products')->group(function () {
        Route::get('/', [ProductController::class, 'index']);
        Route::get('/{product}', [ProductController::class, 'show']);
        Route::post('/', [ProductController::class, 'store']);
        Route::put('/{product}', [ProductController::class, 'update']);
        Route::delete('/{product}', [ProductController::class, 'destroy']);

        // Legacy compatibility
        Route::get('/brand/{brandName}', [ProductController::class, 'byBrandName']);
    });

    Route::prefix('orders')->group(function () {
        Route::get('/', [OrderController::class, 'index']);
        Route::get('/{order}', [OrderController::class, 'show']);
        Route::post('/', [OrderController::class, 'store']);
        Route::put('/{order}/status', [OrderController::class, 'updateStatus']);
        Route::post('/{order}/resend-email', [OrderController::class, 'resendEmail']);
        Route::post('/{order}/regenerate-pdf', [OrderController::class, 'regeneratePDF']);
    });

    // Legacy endpoint for parsing products (compatibility with old frontend)
    Route::get('/parse-products', function () {
        $brands = \App\Models\Brand::active()
            ->ordered()
            ->with(['activeProducts' => function ($q) {
                $q->orderBy('sort_order')->orderBy('name');
            }])
            ->get();

        // Format response to match old CSV parser format
        $result = [];
        foreach ($brands as $brand) {
            $result[$brand->name] = $brand->activeProducts->map(function ($product) {
                return [
                    'name' => $product->name,
                    'size' => $product->size,
                    'barcode' => $product->barcode,
                ];
            })->toArray();
        }

        return response()->json($result);
    });
});

// Health check endpoint
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toISOString(),
        'version' => '1.0.0',
    ]);
});
