# Haleon Orders - Laravel Backend API

Modern Laravel backend for the Haleon pharmaceutical ordering system with queued email/PDF generation, RESTful API, and database-driven product management.

## 🚀 Features

- ✅ RESTful API with versioning (`/api/v1/`)
- ✅ Database-driven product catalog (replaces CSV)
- ✅ Queued email and PDF generation
- ✅ Order management with status tracking
- ✅ Product snapshots for historical accuracy
- ✅ Automated PDF cleanup
- ✅ Legacy API compatibility
- ✅ Comprehensive validation
- ✅ Resource transformers for clean API responses

## 📋 Requirements

- PHP 8.1+
- MySQL 5.7+ or PostgreSQL 12+
- Composer
- Redis (for queues - optional, can use database)
- Node.js & NPM (for asset compilation - optional)

## 🛠️ Installation

### 1. Clone and Install Dependencies

```bash
cd laravel
composer install
```

### 2. Required Packages

Install these Laravel packages:

```bash
composer require barryvdh/laravel-dompdf
composer require laravel/sanctum
```

Optional but recommended:

```bash
composer require spatie/laravel-query-builder
composer require predis/predis
```

### 3. Environment Configuration

```bash
cp .env.example .env
php artisan key:generate
```

Edit `.env` and configure:

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_DATABASE=haleon_orders
DB_USERNAME=your_username
DB_PASSWORD=your_password

MAIL_MAILER=smtp
MAIL_HOST=mail.smtp2go.com
MAIL_PORT=2525
MAIL_USERNAME=auto-order.co.za
MAIL_PASSWORD=Wange@187Sakkie123
MAIL_ENCRYPTION=tls

QUEUE_CONNECTION=database  # or redis for better performance
```

### 4. Database Setup

```bash
# Create database
mysql -u root -p -e "CREATE DATABASE haleon_orders CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

# Run migrations
php artisan migrate

# Import products from CSV
php artisan import:haleon-csv "path/to/Haleon Record Cards.csv"
```

### 5. Storage Setup

```bash
# Create storage link for public file access
php artisan storage:link

# Set permissions
chmod -R 775 storage bootstrap/cache
```

### 6. Queue Worker

For production, run queue workers:

```bash
# Database queues
php artisan queue:work --tries=3

# Redis queues (better performance)
php artisan queue:work redis --tries=3
```

For development, use:

```bash
php artisan queue:listen
```

### 7. Scheduled Tasks

Add to crontab for PDF cleanup:

```bash
* * * * * cd /path/to/laravel && php artisan schedule:run >> /dev/null 2>&1
```

Register in `app/Console/Kernel.php`:

```php
protected function schedule(Schedule $schedule)
{
    $schedule->job(new CleanupExpiredPDFsJob)->daily();
}
```

## 📡 API Endpoints

### Base URL: `/api/v1`

### Brands

```http
GET    /api/v1/brands                    # List all brands
GET    /api/v1/brands?with_products=1    # List brands with products
GET    /api/v1/brands/{id}               # Get brand details
POST   /api/v1/brands                    # Create brand
PUT    /api/v1/brands/{id}               # Update brand
DELETE /api/v1/brands/{id}               # Delete brand
```

### Products

```http
GET    /api/v1/products                  # List all products
GET    /api/v1/products?brand_id=1       # Filter by brand
GET    /api/v1/products?search=keyword   # Search products
GET    /api/v1/products/{id}             # Get product details
POST   /api/v1/products                  # Create product
PUT    /api/v1/products/{id}             # Update product
DELETE /api/v1/products/{id}             # Delete product
```

### Orders

```http
GET    /api/v1/orders                    # List orders
GET    /api/v1/orders?status=completed   # Filter by status
GET    /api/v1/orders/{id}               # Get order details
POST   /api/v1/orders                    # Create order (triggers email/PDF)
PUT    /api/v1/orders/{id}/status        # Update order status
POST   /api/v1/orders/{id}/resend-email  # Resend order email
POST   /api/v1/orders/{id}/regenerate-pdf # Regenerate PDF
```

### Legacy Compatibility

```http
GET /api/v1/parse-products  # Returns products grouped by brand (CSV format)
```

## 📝 API Request Examples

### Create Order

```json
POST /api/v1/orders
Content-Type: application/json

{
  "pharmacy_name": "Main Street Pharmacy",
  "email": "pharmacy@example.com",
  "notes": "Urgent delivery needed",
  "items": [
    {
      "product_id": 1,
      "quantity": 5
    },
    {
      "product_id": 12,
      "quantity": 10
    }
  ]
}
```

### Response

```json
{
  "success": true,
  "message": "Order created successfully. Email will be sent shortly.",
  "data": {
    "id": 1,
    "pharmacy_name": "Main Street Pharmacy",
    "email": "pharmacy@example.com",
    "total_items": 15,
    "status": "pending",
    "created_at": "2024-01-15T10:30:00.000000Z",
    "items": [...]
  }
}
```

## 🗂️ Project Structure

```
app/
├── Console/Commands/
│   └── ImportHaleonCSV.php          # CSV import command
├── Http/
│   ├── Controllers/Api/
│   │   ├── BrandController.php
│   │   ├── ProductController.php
│   │   └── OrderController.php
│   ├── Requests/
│   │   ├── StoreBrandRequest.php
│   │   ├── StoreProductRequest.php
│   │   └── StoreOrderRequest.php
│   └── Resources/
│       ├── BrandResource.php
│       ├── ProductResource.php
│       ├── OrderResource.php
│       ├── OrderItemResource.php
│       └── OrderPdfResource.php
├── Jobs/
│   ├── GeneratePDFJob.php           # PDF generation (queued)
│   ├── SendOrderEmailJob.php        # Email sending (queued)
│   └── CleanupExpiredPDFsJob.php    # Daily PDF cleanup
├── Models/
│   ├── Brand.php
│   ├── Product.php
│   ├── Order.php
│   ├── OrderItem.php
│   └── OrderPdf.php
└── Services/
    ├── PDFService.php               # PDF generation logic
    └── EmailService.php             # Email sending logic

database/migrations/
├── 2024_01_01_000001_create_brands_table.php
├── 2024_01_01_000002_create_products_table.php
├── 2024_01_01_000003_create_orders_table.php
├── 2024_01_01_000004_create_order_items_table.php
└── 2024_01_01_000005_create_order_pdfs_table.php

resources/views/
├── emails/
│   ├── order-confirmation.blade.php
│   └── order-failed.blade.php
└── pdf/
    └── order.blade.php

routes/
└── api.php                          # API routes
```

## 🧪 Testing

```bash
# Run all tests
php artisan test

# Run specific test
php artisan test --filter OrderTest

# With coverage
php artisan test --coverage
```

## 🔧 Artisan Commands

```bash
# Import products from CSV
php artisan import:haleon-csv [file]

# Queue management
php artisan queue:work
php artisan queue:restart
php artisan queue:failed          # List failed jobs
php artisan queue:retry all       # Retry failed jobs

# Database
php artisan migrate:fresh --seed
php artisan db:seed

# Clear cache
php artisan cache:clear
php artisan config:clear
php artisan route:clear
```

## 📊 Database Schema

### brands
- id, name (unique), active, sort_order, timestamps

### products
- id, brand_id, name, size, barcode (unique), active, sort_order, timestamps

### orders
- id, pharmacy_name, email, total_items, status, notes, processed_at, timestamps

### order_items
- id, order_id, product_id, quantity, product_snapshot (JSON), timestamps

### order_pdfs
- id, order_id, file_path, file_name, file_size, expires_at, timestamps

## 🚀 Deployment

### Production Checklist

- [ ] Set `APP_ENV=production` and `APP_DEBUG=false`
- [ ] Set strong `APP_KEY` (auto-generated)
- [ ] Configure production database
- [ ] Set up Redis for queues and cache
- [ ] Configure supervisor for queue workers
- [ ] Set up cron for scheduled tasks
- [ ] Configure HTTPS and SSL certificates
- [ ] Set proper file permissions (755/644)
- [ ] Enable OPcache for PHP
- [ ] Configure log rotation
- [ ] Set up backup strategy
- [ ] Configure monitoring (Laravel Telescope, Sentry)

### Supervisor Configuration

```ini
[program:haleon-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/laravel/artisan queue:work redis --sleep=3 --tries=3
autostart=true
autorestart=true
user=www-data
numprocs=4
redirect_stderr=true
stdout_logfile=/path/to/laravel/storage/logs/worker.log
```

## 🔐 Security Notes

- ⚠️ **IMPORTANT**: Change SMTP credentials in production
- Move sensitive credentials to environment variables
- Enable rate limiting on API endpoints
- Implement authentication (Laravel Sanctum is included)
- Use HTTPS in production
- Regularly update dependencies

## 📈 Performance Tips

- Use Redis for queues and cache
- Enable OPcache and optimize PHP settings
- Index frequently queried database columns
- Use eager loading to prevent N+1 queries
- Implement API rate limiting
- Use CDN for static assets
- Enable Gzip compression

## 🐛 Troubleshooting

### Queue jobs not processing
```bash
php artisan queue:restart
php artisan queue:work --tries=3
```

### PDF generation fails
- Check storage permissions: `chmod -R 775 storage`
- Verify DomPDF is installed: `composer require barryvdh/laravel-dompdf`

### Email not sending
- Verify SMTP credentials in `.env`
- Check logs: `tail -f storage/logs/laravel.log`
- Test email config: `php artisan tinker` → `Mail::raw('Test', fn($msg) => $msg->to('test@example.com')->subject('Test'));`

### Database connection issues
- Verify database credentials
- Check database exists: `SHOW DATABASES;`
- Test connection: `php artisan migrate:status`

## 📞 Support

For issues or questions:
- Check `storage/logs/laravel.log` for errors
- Review queue failed jobs: `php artisan queue:failed`
- Enable debugging: Set `APP_DEBUG=true` in `.env`

## 🎯 Next Steps

- [ ] Implement authentication with Laravel Sanctum
- [ ] Add API documentation (Swagger/Scribe)
- [ ] Write comprehensive tests
- [ ] Set up CI/CD pipeline
- [ ] Implement role-based access control
- [ ] Add API rate limiting
- [ ] Configure monitoring and alerts
